/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Workfile: CifXDeviceChannel.cpp $
   Last Modification:
    $Author: MichaelT $
    $Modtime: 30.10.09 15:14 $
    $Revision: 1353 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
    cifX Channel handler class. Connects and services a channel on a cifX Device
       
   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
      1        28.06.2006  MT       initial version
 
**************************************************************************************/

#include "StdAfx.h"
#include ".\cifxdevicechannel.h"
#include "CifxErrors.h"
#include <string>

///////////////////////////////////////////////////////////////////////////////////////////
/// \file CifXDeviceChannel.cpp
///   cifX Channel handler class. Connects and services a channel on a cifX Device
///////////////////////////////////////////////////////////////////////////////////////////


/////////////////////////////////////////////////////////////////////////////
/// Constructor
///   \param ulChannel      channel number acquired during enumeration
///   \param ptBoardInfo    pointer to Board information structure
///   \param ptChannelInfo  pointer to Channel information structure
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceChannel::CCifXDeviceChannel(uint32_t ulChannel, BOARD_INFORMATION* ptBoardInfo, CHANNEL_INFORMATION* ptChannelInfo)
: CCifXDeviceBase(ptBoardInfo, ptChannelInfo)
, m_hChannel(NULL)
, m_ulChannel(ulChannel)
{
}

/////////////////////////////////////////////////////////////////////////////
/// Destructor
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceChannel::~CCifXDeviceChannel(void)
{
  CloseDevice();
}

/////////////////////////////////////////////////////////////////////////////
/// Checks if the object is a system device (which does not allow I/O, etc)
///   \return false
/////////////////////////////////////////////////////////////////////////////
bool CCifXDeviceChannel::IsSystemDevice(void)
{
  return false;
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Open Device function. Opens a connection to the device via cifX Driver
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::OpenDevice(void)
{
  if(NULL != m_hChannel)
    CloseDevice();

  return xChannelOpen(NULL, m_tBoardInfo.abBoardName, m_ulChannel, &m_hChannel);
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Close Device function. Closes a connection to the device via cifX Driver
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::CloseDevice(void)
{
  int32_t lRet = CIFX_NO_ERROR;

  if(NULL != m_hChannel)
  {
    lRet = xChannelClose(m_hChannel);
    m_hChannel = NULL;
  }

  return lRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Put Packet function. Puts a packet to the device mailbox
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::PutPacket(CIFX_PACKET* ptPacket, uint32_t ulTimeout)
{
  return xChannelPutPacket(m_hChannel, ptPacket, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Overridden Get Packet function. Gets a packet to the device mailbox
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::GetPacket(CIFX_PACKET* ptPacket, uint32_t ulBufferSize, uint32_t ulTimeout)
{
  return xChannelGetPacket(m_hChannel, ulBufferSize, ptPacket, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Queries all properties from the device (used for Open Device dialog)
///   \return Array of displayable properties
/////////////////////////////////////////////////////////////////////////////
std::vector<CCifXDeviceBase::PROPERTY_LIST> CCifXDeviceChannel::GetDescription(void)
{
  std::vector<CCifXDeviceBase::PROPERTY_LIST> cvList(CCifXDeviceBase::GetDescription());

  std::string szFirmwareName;

  szFirmwareName.append( (char*)m_tChannelInfo.abFWName, m_tChannelInfo.bFWNameLength);

  PROPERTY_LIST tList;

  tList.csProperty = _T("Firmware Name");
  tList.csValue    = szFirmwareName.c_str();
  cvList.push_back(tList);

  tList.csProperty = _T("Firmware Version");
  tList.csValue.Format(_T("%u.%u.%u.%u (Build %u)"),
                       m_tChannelInfo.usFWMajor,
                       m_tChannelInfo.usFWMinor,
                       m_tChannelInfo.usFWBuild,
                       m_tChannelInfo.usFWRevision,
                       m_tChannelInfo.usFWBuild);
  cvList.push_back(tList);

  tList.csProperty = _T("Firmware Date");
  tList.csValue.Format(_T("%u/%u/%u"),
                       m_tChannelInfo.bFWMonth,
                       m_tChannelInfo.bFWDay,
                       m_tChannelInfo.usFWYear);
  cvList.push_back(tList);

  return cvList;
}

/////////////////////////////////////////////////////////////////////////////
/// Read I/O area infromation
///   \param ulCmd        Command
///   \param ulAreaNumber Number of the area
///   \param ulOffset     Offset for the read command
///   \param ulDataLen    Length of data to read
///   \param pvData       Pointer to returned data
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::IOInfo(uint32_t ulCmd, uint32_t ulAreaNumber, uint32_t ulSize, void* pvData)
{
  return xChannelIOInfo(m_hChannel, ulCmd, ulAreaNumber, ulSize, pvData);
}

/////////////////////////////////////////////////////////////////////////////
/// Read I/O Command
///   \param ulAreaNumber Number of the area
///   \param ulOffset     Offset for the read command
///   \param ulDataLen    Length of data to read
///   \param pvData       Pointer to returned data
///   \param ulTimeout    Timeout to wait for handshake
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::IORead(uint32_t ulAreaNumber, uint32_t ulOffset, uint32_t ulDataLen, void* pvData, uint32_t ulTimeout)
{
  return xChannelIORead(m_hChannel, ulAreaNumber, ulOffset, ulDataLen, pvData, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Write I/O Command
///   \param ulAreaNumber Number of the area
///   \param ulOffset     Offset for the read command
///   \param ulDataLen    Length of send data
///   \param pvData       Pointer send data
///   \param ulTimeout    Timeout to wait for handshake
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::IOWrite(uint32_t ulAreaNumber, uint32_t ulOffset, uint32_t ulDataLen, void* pvData, uint32_t ulTimeout)
{
  return xChannelIOWrite(m_hChannel, ulAreaNumber, ulOffset, ulDataLen, pvData, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Read back I/O Output data
///   \param ulAreaNumber Number of the area
///   \param ulOffset     Offset for the read command
///   \param ulDataLen    Length of send data
///   \param pvData       Pointer send data
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::IOReadSend(uint32_t ulAreaNumber, uint32_t ulOffset, uint32_t ulDataLen, void* pvData)
{
  return xChannelIOReadSendData(m_hChannel, ulAreaNumber, ulOffset, ulDataLen, pvData);
}

/////////////////////////////////////////////////////////////////////////////
/// Get Channel's Property/Value pairs for the Channel/Device
///   \return Property/Value pairs (strings)
/////////////////////////////////////////////////////////////////////////////
CCifXDeviceBase::PROPERTY_VECTOR CCifXDeviceChannel::GetChannelInformation(uint32_t ulCmd)
{
  PROPERTY_VECTOR     cvRet;

  switch(ulCmd)
  {
  case CIFX_INFO_CHANNELIO:
    {
      CHANNEL_INFORMATION tChannelInfo = {0};
      int32_t lRet = xChannelInfo(m_hChannel, 
                               sizeof(tChannelInfo), 
                               &tChannelInfo);
                               
      if( CIFX_NO_ERROR == lRet)
      {    
        PROPERTY_LIST tList;
        for ( uint32_t ulIdx = 0; ulIdx < tChannelInfo.ulIOInAreaCnt; ulIdx++)
        {
          CHANNEL_IO_INFORMATION  tIOInfo = {0};

          if( xChannelIOInfo( m_hChannel,
                              CIFX_IO_INPUT_AREA,
                              ulIdx, 
                              sizeof(tIOInfo), 
                              &tIOInfo) == CIFX_NO_ERROR)
          {
            tList.csProperty.Format(_T("I/O Input Area %u:"), ulIdx);
            tList.csValue = _T("--------------------------------");
            cvRet.push_back(tList);
          
            tList.csProperty = _T("Size");
            tList.csValue.Format(_T("%u"), tIOInfo.ulTotalSize);
            cvRet.push_back(tList);
           
            tList.csProperty = _T("I/O Mode");
            tList.csValue.Format(_T("%u"), tIOInfo.ulIOMode);
            cvRet.push_back(tList);
            
          }
        }
            
        for ( uint32_t ulIdx = 0; ulIdx < tChannelInfo.ulIOOutAreaCnt; ulIdx++)
        {
          CHANNEL_IO_INFORMATION  tIOInfo = {0};

          if( xChannelIOInfo( m_hChannel,
                              CIFX_IO_OUTPUT_AREA,
                              ulIdx,
                              sizeof(tIOInfo), 
                              &tIOInfo) == CIFX_NO_ERROR)
          {
            tList.csProperty.Format(_T("I/O Output Area %u:"), ulIdx);
            tList.csValue = _T("--------------------------------");
            cvRet.push_back(tList);
          
            tList.csProperty = _T("Size");
            tList.csValue.Format(_T("%u"), tIOInfo.ulTotalSize);
            cvRet.push_back(tList);
           
            tList.csProperty = _T("Mode");
            tList.csValue.Format(_T("%u"), tIOInfo.ulIOMode);
            cvRet.push_back(tList);
          }
        }
      }
    }
    break;

  case CIFX_INFO_CHANNEL:
  default:
    {
      CHANNEL_INFORMATION tChannelInfo = {0};
      int32_t lRet = xChannelInfo(m_hChannel, 
                              sizeof(tChannelInfo), 
                              &tChannelInfo);

      if(CIFX_NO_ERROR == lRet)
      {
        PROPERTY_LIST tList;

        tList.csProperty = _T("Board Name");
        tList.csValue    = tChannelInfo.abBoardName;
        cvRet.push_back(tList);

        tList.csProperty = _T("Board Alias");
        tList.csValue    = tChannelInfo.abBoardAlias;
        cvRet.push_back(tList);

        tList.csProperty = _T("Device Number");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulDeviceNumber);
        cvRet.push_back(tList);

        tList.csProperty = _T("Serial Number");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulSerialNumber);
        cvRet.push_back(tList);

        std::string szFirmwareName;

        szFirmwareName.append( (char*)m_tChannelInfo.abFWName, m_tChannelInfo.bFWNameLength);

        tList.csProperty = _T("Firmware Name");
        tList.csValue    = szFirmwareName.c_str();
        cvRet.push_back(tList);

        tList.csProperty = _T("Firmware Version");
        tList.csValue.Format(_T("%u.%u.%u.%u (Build %u)"),
                            m_tChannelInfo.usFWMajor,
                            m_tChannelInfo.usFWMinor,                            
                            m_tChannelInfo.usFWBuild,
                            m_tChannelInfo.usFWRevision,
                            m_tChannelInfo.usFWBuild);
        cvRet.push_back(tList);

        tList.csProperty = _T("Firmware Date");
        tList.csValue.Format(_T("%u/%u/%u"),
                            m_tChannelInfo.bFWMonth,
                            m_tChannelInfo.bFWDay,
                            m_tChannelInfo.usFWYear);
        cvRet.push_back(tList);

        tList.csProperty = _T("Channel Error");
        tList.csValue.Format(_T("0x%08X"), tChannelInfo.ulChannelError);
        cvRet.push_back(tList);

        tList.csProperty = _T("Open Count");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulOpenCnt);
        cvRet.push_back(tList);

        tList.csProperty = _T("Put Packet Count");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulPutPacketCnt);
        cvRet.push_back(tList);

        tList.csProperty = _T("Get Packet Count");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulGetPacketCnt);
        cvRet.push_back(tList);

        tList.csProperty = _T("Mailbox Size");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulMailboxSize);
        cvRet.push_back(tList);

        tList.csProperty = _T("I/O Input Area Count");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulIOInAreaCnt);
        cvRet.push_back(tList);

        tList.csProperty = _T("I/O Output Area Count");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulIOOutAreaCnt);
        cvRet.push_back(tList);

        tList.csProperty = _T("Handshake Size");
        tList.csValue.Format(_T("%u"), tChannelInfo.ulHskSize);
        cvRet.push_back(tList);

        tList.csProperty = _T("netX Status Flags");
        tList.csValue.Format(_T("0x%08X"), tChannelInfo.ulNetxFlags);
        cvRet.push_back(tList);

        tList.csProperty = _T("Host Status Flags");
        tList.csValue.Format(_T("0x%08X"), tChannelInfo.ulHostFlags);
        cvRet.push_back(tList);

        tList.csProperty = _T("Host COS Flags");
        tList.csValue.Format(_T("0x%08X"), tChannelInfo.ulHostCOSFlags);
        cvRet.push_back(tList);

        tList.csProperty = _T("Device COS Flags");
        tList.csValue.Format(_T("0x%08X"), tChannelInfo.ulDeviceCOSFlags);
        cvRet.push_back(tList);
      }
    }
    break;
  }

  return cvRet;
}

/////////////////////////////////////////////////////////////////////////////
/// Query the Name of the supplied channel information
///   \return Channel information description
/////////////////////////////////////////////////////////////////////////////
CString CCifXDeviceChannel::GetChannelInfoName(void)
{
  return _T("Channel Information");
}

/////////////////////////////////////////////////////////////////////////////
/// Trigger Channel/Device Watchdog
///   \param ulCmd      Start/Stop command
///   \param pulTrigger Returned watchdog cell value
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::Watchdog(uint32_t ulCmd, uint32_t* pulTrigger)
{
  return xChannelWatchdog(m_hChannel, ulCmd, pulTrigger);
}

/////////////////////////////////////////////////////////////////////////////
/// Read/Write Host State
///   \param ulCmd      Read/Write command
///   \param pulState   Returned state on read commands
///   \param ulTimeout  Timeout to wait for Communication on Host_Ready
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::HostState(uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout)
{
  return xChannelHostState(m_hChannel, ulCmd, pulState, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Read/Write Bus State
///   \param ulCmd      Read/Write command
///   \param pulState   Returned state on read commands
///   \param ulTimeout  Timeout to wait for Communication on Host_Ready
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::BusState(uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout)
{
  return xChannelBusState(m_hChannel, ulCmd, pulState, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Read/Write DMA State
///   \param ulCmd      Read/Write command
///   \param pulState   Returned state on read commands
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::DMAState(uint32_t ulCmd, uint32_t* pulState)
{
  return xChannelDMAState(m_hChannel, ulCmd, pulState);
}

/////////////////////////////////////////////////////////////////////////////
/// Download a file to the Channel/Device
///   \param ulChannel    Channel number
///   \param ulMode       Download Mode
///   \param szFileName   Short file name
///   \param pabFileData  File content
///   \param ulFileSize   Size of the file
///   \param pfnCallback  Progress Callback
///   \param pvUser       User parameter to pass to callback
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::Download(uint32_t /*ulChannel*/, uint32_t ulMode, char* szFileName, 
                                  unsigned char* pabFileData, uint32_t ulFileSize, 
                                  PFN_PROGRESS_CALLBACK pfnCallback, void* pvUser)
{
  return xChannelDownload(m_hChannel, ulMode, szFileName, pabFileData, ulFileSize, pfnCallback, NULL, pvUser);
}

/////////////////////////////////////////////////////////////////////////////
/// Get the current mailbox fill states
///   \param pulRecvCount Number of packet waiting to be received on device
///   \param pulSendCount Number of packets that can be send to the device
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::GetMBXState(uint32_t* pulRecvCount, uint32_t* pulSendCount)
{
  return xChannelGetMBXState(m_hChannel, pulRecvCount, pulSendCount);
}

/////////////////////////////////////////////////////////////////////////////
/// Reset the channel
///   \param ulMode     Reset mode
///   \param ulTimeout  Timeout for reset to be performed
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::Reset(uint32_t ulMode, uint32_t ulTimeout)
{
  return xChannelReset(m_hChannel, ulMode, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Set/Remove/Get Configuration lock state
///   \param ulCmd      ConfigLock Command (CIFX_CONFIGURATION_XXX defines)
///   \param pulState   Returned state if CIFX_CONFIGURATION_GETLCOKSTATE is used as command
///   \param ulTimeout  Timeout in ms for waiting for card to acknowledge lock
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::ConfigLock(uint32_t ulCmd, uint32_t* pulState, uint32_t ulTimeout)
{
  return xChannelConfigLock(m_hChannel, ulCmd, pulState, ulTimeout);
}

/////////////////////////////////////////////////////////////////////////////
/// Finds the first file entry
///   \param ulChannel        Channel number
///   \param ptDirectoryInfo  Pointer to directory info
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::FindFirstFile(uint32_t /*ulChannel*/, CIFX_DIRECTORYENTRY* ptDirectoryInfo)
{
  return xChannelFindFirstFile(m_hChannel, ptDirectoryInfo, NULL, NULL);
}

/////////////////////////////////////////////////////////////////////////////
/// Finds the next file entry
///   \param ulChannel        Channel number
///   \param ptDirectoryInfo  Pointer to directory info
///   \return CIFX_NO_ERROR on success
/////////////////////////////////////////////////////////////////////////////
int32_t CCifXDeviceChannel::FindNextFile(uint32_t /*ulChannel*/, CIFX_DIRECTORYENTRY* ptDirectoryInfo)
{
  return xChannelFindNextFile(m_hChannel, ptDirectoryInfo, NULL, NULL);
}

int32_t CCifXDeviceChannel::Upload(char* szFilename, uint32_t /*ulChannel*/, unsigned char* pbBuffer, uint32_t* pulFileSize)
{
  return xChannelUpload(m_hChannel, DOWNLOAD_MODE_FILE, szFilename,  
                        pbBuffer, pulFileSize, NULL, NULL, NULL);
}
